<?php

if (!defined('BASEPATH')) {
    exit('No direct script access allowed');
}

class Webhooks extends CI_Controller {

    public function __construct()
    {
        parent::__construct();
        $this->load->model('paymentsetting_model');
        $this->load->model('course_model');
    }

    public function insta_webhook() {
        $insta_webhook = $this->paymentsetting_model->insta_webhooksalt();

        $data = $_POST;
        $mac_provided = $data['mac'];  // Get the MAC from the POST data
        unset($data['mac']);  // Remove the MAC key from the data.
        $ver = explode('.', phpversion());
        $major = (int) $ver[0];
        $minor = (int) $ver[1];
        if ($major >= 5 and $minor >= 4) {
            ksort($data, SORT_STRING | SORT_FLAG_CASE);
        } else {
            uksort($data, 'strcasecmp');
        }
// You can get the 'salt' from Instamojo's developers page(make sure to log in first): https://www.instamojo.com/developers
// Pass the 'salt' without <>
        $mac_calculated = hash_hmac("sha1", implode("|", $data), "$insta_webhook->salt");
        if ($mac_provided == $mac_calculated) {
            if ($data['status'] == "Credit") {
                // Payment was successful, mark it as successful in your database.
                // You can acess payment_request_id, purpose etc here. 
            } else {
                // Payment was unsuccessful, mark it as failed in your database.
                // You can acess payment_request_id, purpose etc here.
            }
        } else {
            echo "MAC mismatch";
        }
    }

    public function cashfree() {
        // Log incoming webhook data with timestamp
        $input = file_get_contents('php://input');
        $timestamp = date('Y-m-d H:i:s');
        log_message('info', "[{$timestamp}] Cashfree Webhook received: " . $input);
        
        // Log headers for debugging
        $headers = getallheaders();
        log_message('info', "[{$timestamp}] Cashfree Webhook headers: " . json_encode($headers));
        
        // Parse JSON data
        $data = json_decode($input, true);
        
        if (!$data) {
            log_message('error', "[{$timestamp}] Cashfree Webhook: Invalid JSON data - Raw input: " . $input);
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'Invalid JSON']);
            return;
        }
        
        // Log parsed data structure
        log_message('info', "[{$timestamp}] Cashfree Webhook parsed data: " . json_encode($data));
        
        // Verify webhook signature
        $signature = isset($headers['x-webhook-signature']) ? $headers['x-webhook-signature'] : 
                    (isset($headers['X-Webhook-Signature']) ? $headers['X-Webhook-Signature'] : '');
        
        log_message('info', "[{$timestamp}] Cashfree Webhook signature: " . $signature);
        
        if (!$this->verifySignature($input, $signature)) {
            log_message('error', "[{$timestamp}] Cashfree Webhook: Invalid signature - Expected vs Received");
            // Continue processing even if signature fails for debugging mobile payments
            log_message('warning', "[{$timestamp}] Cashfree Webhook: Continuing despite signature failure for debugging");
        }
        
        // Extract payment information with multiple fallback paths
        $orderId = $this->extractOrderId($data);
        $orderStatus = $this->extractOrderStatus($data);
        $paymentId = $this->extractPaymentId($data);
        
        log_message('info', "[{$timestamp}] Extracted - OrderID: {$orderId}, Status: {$orderStatus}, PaymentID: {$paymentId}");
        
        if (!$orderId || !$orderStatus) {
            log_message('error', "[{$timestamp}] Cashfree Webhook: Missing required data - orderId: {$orderId}, orderStatus: {$orderStatus}");
            log_message('error', "[{$timestamp}] Full data structure: " . json_encode($data));
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'Missing required data']);
            return;
        }
        
        log_message('info', "[{$timestamp}] Cashfree Webhook: Processing order {$orderId} with status {$orderStatus}");
        
        try {
            if (strtoupper($orderStatus) === 'SUCCESS' || strtoupper($orderStatus) === 'PAID') {
                $result = $this->processSuccessfulPayment($data);
                log_message('info', "[{$timestamp}] Success payment processing result: " . json_encode($result));
            } elseif (strtoupper($orderStatus) === 'FAILED' || strtoupper($orderStatus) === 'CANCELLED') {
                $result = $this->processFailedPayment($data);
                log_message('info', "[{$timestamp}] Failed payment processing result: " . json_encode($result));
            } else {
                log_message('info', "[{$timestamp}] Cashfree Webhook: Unhandled status {$orderStatus} for order {$orderId}");
            }
            
            http_response_code(200);
            echo json_encode(['status' => 'success', 'message' => 'Webhook processed', 'timestamp' => $timestamp]);
            
        } catch (Exception $e) {
            log_message('error', "[{$timestamp}] Cashfree Webhook Error: " . $e->getMessage());
            log_message('error', "[{$timestamp}] Error trace: " . $e->getTraceAsString());
            http_response_code(500);
            echo json_encode(['status' => 'error', 'message' => 'Internal server error', 'timestamp' => $timestamp]);
        }
    }
    
    private function processSuccessfulPayment($data) {
        $timestamp = date('Y-m-d H:i:s');
        $orderId = $this->extractOrderId($data);
        $paymentId = $this->extractPaymentId($data);
        $orderAmount = $this->extractOrderAmount($data);
        
        log_message('info', "[{$timestamp}] Processing successful payment - OrderID: {$orderId}, PaymentID: {$paymentId}, Amount: {$orderAmount}");
        
        // Extract customer ID from data (format: "Student_id_123")
        $customerId = $this->extractCustomerId($data);
        $studentId = null;
        if ($customerId && preg_match('/Student_id_(\d+)/', $customerId, $matches)) {
            $studentId = $matches[1];
            log_message('info', "[{$timestamp}] Extracted student ID: {$studentId} from customer ID: {$customerId}");
        }
        
        // First try to get course payment record by transaction_id
        $coursePayment = $this->course_model->getByTransactionId($orderId);
        
        if ($coursePayment) {
            log_message('info', "[{$timestamp}] Found existing course payment record: " . json_encode($coursePayment));
            
            // Update payment status
            $updateData = [
                'payment_status' => 'SUCCESS',
                'updated_at' => date('Y-m-d H:i:s'),
                'cf_payment_id' => $paymentId
            ];
            
            $affected = $this->course_model->updatePayment($orderId, $updateData);
            log_message('info', "[{$timestamp}] Course payment updated for order: {$orderId}, affected rows: {$affected}");
            
            return ['success' => true, 'message' => 'Payment updated successfully', 'affected_rows' => $affected];
        } else {
            log_message('warning', "[{$timestamp}] Course payment not found for order: {$orderId}");
            
            // If no existing record found and we have student ID, create a new payment record
            if ($studentId && $orderAmount) {
                log_message('info', "[{$timestamp}] Creating new payment record for student: {$studentId}");
                
                // Get student and course details from processing payment table
                $processingPayment = $this->getProcessingPaymentByStudentId($studentId);
                
                if ($processingPayment) {
                    $paymentData = [
                        'date' => date('Y-m-d'),
                        'student_id' => $studentId,
                        'online_courses_id' => $processingPayment->online_courses_id,
                        'course_name' => $processingPayment->course_name,
                        'actual_price' => $processingPayment->actual_price,
                        'paid_amount' => $orderAmount,
                        'payment_type' => 'Online',
                        'transaction_id' => $orderId,
                        'note' => "Online course fees deposit through Cashfree Webhook - Order: {$orderId}, Payment: {$paymentId}",
                        'payment_mode' => 'Cashfree',
                        'payment_status' => 'SUCCESS',
                        'cf_payment_id' => $paymentId,
                        'created_at' => date('Y-m-d H:i:s'),
                        'updated_at' => date('Y-m-d H:i:s')
                    ];
                    
                    $insertId = $this->course_model->add($paymentData);
                    log_message('info', "[{$timestamp}] New payment record created with ID: {$insertId}");
                    
                    // Clean up processing payment record
                    $this->deleteProcessingPayment($processingPayment->id);
                    
                    return ['success' => true, 'message' => 'New payment record created', 'insert_id' => $insertId];
                } else {
                    log_message('error', "[{$timestamp}] No processing payment found for student: {$studentId}");
                }
            }
            
            return ['success' => false, 'message' => 'Course payment record not found and cannot create new record'];
        }
    }
    
    private function processFailedPayment($data) {
        $timestamp = date('Y-m-d H:i:s');
        $orderId = $this->extractOrderId($data);
        $failureReason = $this->extractFailureReason($data);
        
        log_message('info', "[{$timestamp}] Processing failed payment - OrderID: {$orderId}, Reason: {$failureReason}");
        
        // Get course payment record
        $coursePayment = $this->course_model->getByTransactionId($orderId);
        
        if ($coursePayment) {
            log_message('info', "[{$timestamp}] Found course payment record for failed payment: " . json_encode($coursePayment));
            
            // Update payment status
            $updateData = [
                'payment_status' => 'FAILED',
                'failure_reason' => $failureReason,
                'updated_at' => date('Y-m-d H:i:s')
            ];
            
            $affected = $this->course_model->updatePayment($orderId, $updateData);
            log_message('info', "[{$timestamp}] Course payment failed for order: {$orderId}, reason: {$failureReason}, affected rows: {$affected}");
            
            return ['success' => true, 'message' => 'Failed payment updated', 'affected_rows' => $affected];
        } else {
            log_message('warning', "[{$timestamp}] Course payment not found for failed order: {$orderId}");
            return ['success' => false, 'message' => 'Course payment record not found for failed payment'];
        }
    }
    
    private function extractCustomerId($data) {
        // Try different paths for customer ID
        if (isset($data['data']['customer']['customer_id'])) {
            return $data['data']['customer']['customer_id'];
        }
        if (isset($data['customer']['customer_id'])) {
            return $data['customer']['customer_id'];
        }
        if (isset($data['data']['order']['customer_details']['customer_id'])) {
            return $data['data']['order']['customer_details']['customer_id'];
        }
        return null;
    }
    
    private function extractFailureReason($data) {
        // Try different paths for failure reason
        if (isset($data['data']['payment']['payment_message'])) {
            return $data['data']['payment']['payment_message'];
        }
        if (isset($data['data']['order']['order_note'])) {
            return $data['data']['order']['order_note'];
        }
        if (isset($data['payment_message'])) {
            return $data['payment_message'];
        }
        if (isset($data['failure_reason'])) {
            return $data['failure_reason'];
        }
        return 'Payment failed - no specific reason provided';
    }
    
    private function extractOrderAmount($data) {
        // Try different paths for order amount
        if (isset($data['data']['order']['order_amount'])) {
            return $data['data']['order']['order_amount'];
        }
        if (isset($data['order']['order_amount'])) {
            return $data['order']['order_amount'];
        }
        if (isset($data['data']['payment']['payment_amount'])) {
            return $data['data']['payment']['payment_amount'];
        }
        return null;
    }
    
    private function getProcessingPaymentByStudentId($studentId) {
        $this->db->select('*');
        $this->db->from('online_course_processing_payment');
        $this->db->where('student_id', $studentId);
        $this->db->order_by('id', 'DESC');
        $this->db->limit(1);
        $query = $this->db->get();
        return $query->row();
    }
    
    private function deleteProcessingPayment($id) {
        $this->db->where('id', $id);
        $this->db->delete('online_course_processing_payment');
        return $this->db->affected_rows();
    }
    
    private function verifySignature($payload, $signature) {
        $timestamp = date('Y-m-d H:i:s');
        
        // Load Cashfree configuration
        $this->load->model('paymentsetting_model');
        $config = $this->paymentsetting_model->getActiveMethod();
        
        if (!$config || !isset($config->api_secret_key)) {
            log_message('error', "[{$timestamp}] Cashfree API secret key not found in config");
            return false;
        }
        
        log_message('info', "[{$timestamp}] Verifying signature with secret key length: " . strlen($config->api_secret_key));
        
        $secretKey = $config->api_secret_key;
        
        // Cashfree v2023-08-01 uses timestamp + raw body for signature
        // Extract timestamp from signature header (format: "t=timestamp,v1=signature")
        $signatureParts = [];
        if (preg_match_all('/(\w+)=([^,]+)/', $signature, $matches, PREG_SET_ORDER)) {
            foreach ($matches as $match) {
                $signatureParts[$match[1]] = $match[2];
            }
        }
        
        if (!isset($signatureParts['t']) || !isset($signatureParts['v1'])) {
            log_message('error', "[{$timestamp}] Invalid signature format: {$signature}");
            return false;
        }
        
        $timestamp_header = $signatureParts['t'];
        $received_signature = $signatureParts['v1'];
        
        // Create signed payload: timestamp + "." + raw body
        $signed_payload = $timestamp_header . "." . $payload;
        
        // Calculate expected signature using HMAC-SHA256
        $expectedSignature = hash_hmac('sha256', $signed_payload, $secretKey);
        
        log_message('info', "[{$timestamp}] Signature verification - Expected: {$expectedSignature}, Received: {$received_signature}");
        log_message('info', "[{$timestamp}] Signed payload: {$signed_payload}");
        
        return hash_equals($expectedSignature, $received_signature);
    }
    
    private function extractOrderId($data) {
        // Try multiple paths to extract order ID
        if (isset($data['data']['order']['order_id'])) {
            return $data['data']['order']['order_id'];
        }
        if (isset($data['order_id'])) {
            return $data['order_id'];
        }
        if (isset($data['data']['order_id'])) {
            return $data['data']['order_id'];
        }
        if (isset($data['order']['order_id'])) {
            return $data['order']['order_id'];
        }
        return null;
    }
    
    private function extractOrderStatus($data) {
        // Try multiple paths to extract order status
        if (isset($data['data']['payment']['payment_status'])) {
            return $data['data']['payment']['payment_status'];
        }
        if (isset($data['data']['order_status'])) {
            return $data['data']['order_status'];
        }
        if (isset($data['order_status'])) {
            return $data['order_status'];
        }
        if (isset($data['order']['order_status'])) {
            return $data['order']['order_status'];
        }
        // Also check payment status as fallback
        if (isset($data['data']['payment']['payment_status'])) {
            return $data['data']['payment']['payment_status'];
        }
        return null;
    }
    
    private function extractPaymentId($data) {
        // Try multiple paths to extract payment ID
        if (isset($data['data']['cf_payment_id'])) {
            return $data['data']['cf_payment_id'];
        }
        if (isset($data['cf_payment_id'])) {
            return $data['cf_payment_id'];
        }
        if (isset($data['data']['payment']['payment_id'])) {
            return $data['data']['payment']['payment_id'];
        }
        if (isset($data['payment_id'])) {
            return $data['payment_id'];
        }
        return null;
    }
    
    private function extractFailureReason($data) {
        // Try multiple paths to extract failure reason
        if (isset($data['data']['payment']['payment_message'])) {
            return $data['data']['payment']['payment_message'];
        }
        if (isset($data['payment_message'])) {
            return $data['payment_message'];
        }
        if (isset($data['data']['payment']['failure_reason'])) {
            return $data['data']['payment']['failure_reason'];
        }
        return 'Payment failed - no specific reason provided';
    }
    
    private function tryAlternativeOrderLookup($orderId, $timestamp) {
        log_message('info', "[{$timestamp}] Trying alternative order lookup methods for: {$orderId}");
        
        // Try to find all course payments to debug
        $this->db->select('*');
        $this->db->from('online_course_payment');
        $this->db->order_by('id', 'DESC');
        $this->db->limit(10);
        $recent_payments = $this->db->get()->result();
        
        log_message('info', "[{$timestamp}] Recent 10 course payments: " . json_encode($recent_payments));
        
        // Try partial match
        $this->db->select('*');
        $this->db->from('online_course_payment');
        $this->db->like('transaction_id', $orderId);
        $partial_match = $this->db->get()->result();
        
        if (!empty($partial_match)) {
            log_message('info', "[{$timestamp}] Found partial matches: " . json_encode($partial_match));
        }
    }

}
